//===-- Common utils for exp10f16 -------------------------------*- C++ -*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIBC_SRC___SUPPORT_MATH_EXP10F16_UTILS_H
#define LLVM_LIBC_SRC___SUPPORT_MATH_EXP10F16_UTILS_H

#include "include/llvm-libc-macros/float16-macros.h"

#ifdef LIBC_TYPES_HAS_FLOAT16

#include "exp10_float16_constants.h"
#include "expf16_utils.h"
#include "src/__support/FPUtil/FPBits.h"

namespace LIBC_NAMESPACE_DECL {

LIBC_INLINE static ExpRangeReduction exp10_range_reduction(float16 x) {
  // For -8 < x < 5, to compute 10^x, we perform the following range reduction:
  // find hi, mid, lo, such that:
  //   x = (hi + mid) * log2(10) + lo, in which
  //     hi is an integer,
  //     mid * 2^3 is an integer,
  //     -2^(-4) <= lo < 2^(-4).
  // In particular,
  //   hi + mid = round(x * 2^3) * 2^(-3).
  // Then,
  //   10^x = 10^(hi + mid + lo) = 2^((hi + mid) * log2(10)) + 10^lo
  // We store 2^mid in the lookup table EXP2_MID_BITS, and compute 2^hi * 2^mid
  // by adding hi to the exponent field of 2^mid.  10^lo is computed using a
  // degree-4 minimax polynomial generated by Sollya.

  float xf = x;
  float kf = fputil::nearest_integer(xf * (LOG2F_10 * 0x1.0p+3f));
  int x_hi_mid = static_cast<int>(kf);
  unsigned x_hi = static_cast<unsigned>(x_hi_mid) >> 3;
  unsigned x_mid = static_cast<unsigned>(x_hi_mid) & 0x7;
  // lo = x - (hi + mid) = round(x * 2^3 * log2(10)) * log10(2) * (-2^(-3)) + x
  float lo = fputil::multiply_add(kf, LOG10F_2 * -0x1.0p-3f, xf);

  uint32_t exp2_hi_mid_bits =
      EXP2_MID_BITS[x_mid] +
      static_cast<uint32_t>(x_hi << fputil::FPBits<float>::FRACTION_LEN);
  float exp2_hi_mid = fputil::FPBits<float>(exp2_hi_mid_bits).get_val();
  // Degree-4 minimax polynomial generated by Sollya with the following
  // commands:
  //   > display = hexadecimal;
  //   > P = fpminimax((10^x - 1)/x, 3, [|SG...|], [-2^-4, 2^-4]);
  //   > 1 + x * P;
  float exp10_lo = fputil::polyeval(lo, 0x1p+0f, 0x1.26bb14p+1f, 0x1.53526p+1f,
                                    0x1.04b434p+1f, 0x1.2bcf9ep+0f);
  return {exp2_hi_mid, exp10_lo};
}

} // namespace LIBC_NAMESPACE_DECL

#endif // LIBC_TYPES_HAS_FLOAT16

#endif // LLVM_LIBC_SRC___SUPPORT_MATH_EXP10F16_UTILS_H
